//*****************************************************************************
//*****************************************************************************
// Implementation file for header "DLA.h"
//*****************************************************************************
//*****************************************************************************
// Phillip Forkner
// Sep. 25, 2002
// See header file for program description
//*****************************************************************************

#include "DLA.h"
#include <math.h>
#include <iostream>
#include <iomanip>
#include <fstream>

#include "ran3.cpp"	// Added 10-18-02

using namespace std;

//-----------------------------------------------------------------------------
// Default constructor
// Sets the initial values
// Should check for improper values
DLA::DLA(int gridSize, int deltaSpawn, int deltaKill,
		 double pot, int modV, double b)
{
	// initialize variables
	r = 0;
	particles = 1;
	n = gridSize;
	dSpawn = deltaSpawn;
	dKill = deltaKill;
	potential = pot;
	modVal = modV;
	beta = b;
}

//-----------------------------------------------------------------------------
// Destructor
DLA::~DLA()
{}

// ----------------------------------------------------------------------------
// Runs the simulation
void DLA::simulate()
{
	initGrid();
	int i = 01;			// counter
	while (getParticle())
	{
		i++;
		while ( !checkStick() && moveCartesian() ){}
		if ( i%modVal == 0 )
		{
			cout << "iteration # = " << i << ", number of particles = " << particles << "\n";
		}
	}
	iterations = i;
	cout << "Total number of particles: " << particles << "\n";
}

// ----------------------------------------------------------------------------
// Writes just the potentials to file (row by row)
void DLA::potentialValsOut()
{
	ofstream outFile("Pot.dat");

	outFile << "Created: Dec 20, 2002 by DLA.cpp \n";
	outFile << "Grid size, n = " << n << "\n";
	outFile << "Iterations (particles introduced) = " << iterations << "\n";
	outFile << "Number of particles stuck= " << particles << "\n";
	outFile << "dSpawn = " << dSpawn << "\n";
	outFile << "dKill = " << dKill << "\n";
	outFile << "potential = " << potential << "\n";
	outFile << "beta = " << beta << "\n";
	outFile << "ran3.cpp seeded with 235678 \n";
	for (int y=0; y<n; y++)
		for (int x=0; x<n; x++)
			outFile << grid[y][x] << "\n";

	outFile.close();
}

// ----------------------------------------------------------------------------
// Writes particle points to file (x,y)
void DLA::gridOut()
{
	ofstream outFile("Grid.dat");

	outFile << "Created: Dec 20, 2002 by DLA.cpp \n";
	outFile << "Grid size, n = " << n << "\n";
	outFile << "Iterations (particles introduced) = " << iterations << "\n";
	outFile << "Number of particles stuck = " << particles << "\n";
	outFile << "dSpawn = " << dSpawn << "\n";
	outFile << "dKill = " << dKill << "\n";
	outFile << "potential = " << potential << "\n";
	outFile << "beta = " << beta << "\n";
	outFile << "ran3.cpp seeded with 235678 \n";
	for (int y=0; y<n; y++)
		for (int x=0; x<n; x++)
			if (grid[y][x] == potential) outFile << x << " " << y << "\n";

	outFile.close();
}

// ----------------------------------------------------------------------------
// Initializes the grid to zeros with a seed in the center. The value of seed
// is equal to potential.
void DLA::initGrid()
{
	for (int i=0; i<n; i++)
		for (int j=0; j<n; j++)
			grid[i][j] = 0;
	grid[n/2][n/2] = potential;
	
	// intialize the grid with a cross in the center
	for (i=-5; i<=5; i++)
	{
		grid[n/2][n/2-i] = potential;
		grid[n/2-i][n/2] = potential;
	}
}

// ----------------------------------------------------------------------------
// gives a random number in [0,n].
float DLA::getRandy(float n)
{
	long seed = 235678;
	float z = ran3(&seed)*n;
	return z;
}

// ----------------------------------------------------------------------------
// introduces a particle on a circle of radius, r+dSpawn.
bool DLA::getParticle()
{
	double pi = 3.141592654;
	float theta = getRandy(359)*2*pi/360;
	if (r+dSpawn < n/2-1)
	{
		x = (n/2)+(r+dSpawn)*cos(theta);
		y = (n/2)+(r+dSpawn)*sin(theta);
		return true;
	}
	else
	{
		cout << "Growth reached the edge \n";
		return false;
	}
}
// ----------------------------------------------------------------------------
// Move the particle in any of four directions (up, down, left, right)
// Returns true if the particle was moved successfully, false, otherwise.
// Assumes that the particle has no existing nearest-neighbors
bool DLA::moveCartesian()
{
	// get a direction
	int direction = findProba();

	if (direction == 1)
		y -= 1;	// move up
	else if (direction == 2)
		y += 1;	// move down
	else if (direction == 3)
		x -= 1;	// move left
	else if (direction == 4)
		x += 1;	// move right

	// Display the grid
	/*
	int temp;
	for (int j=0;j<n;j++)
	{
		for (int i=0;i<n;i++)
			if ( (i==x) && (j==y) )
				cout << setw(1) << potential;
			else
			{
				temp = grid[j][i];
				cout << setw(1) << temp;
			}
		cout << "\n";
	}

	// pause
	for (int i=0;i<50000000;i++){}
	*/

	// check bounds
	// Will return false if the particle is on the edge
	double magnitude = sqrt((x-n/2)*(x-n/2) + (y-n/2)*(y-n/2));

	if ((x <= 0) || (x >= n-1))		// X out of bounds
		return false;
	else if ((y <= 0) || (y >= n-1))// Y out of bounds
		return false;
	else if (magnitude > r+dKill)	// Reached kill radius
	{
		//cout << "Killed \n";
		return false;
	}
	else return true;
}

// ----------------------------------------------------------------------------
// Check to see if the particle sticks
bool DLA::checkStick()
{
	// Check for nearest neighbor
	if ((grid[y][x-1] == potential) || (grid[y][x+1] == potential) || 
		(grid[y-1][x] == potential) || (grid[y+1][x] == potential))
	{
		grid[y][x] = potential;
		double magnitude = sqrt((x-n/2)*(x-n/2) + (y-n/2)*(y-n/2));
		if (magnitude > r) r = magnitude;
		particles += 1;
		calcPot();
		return true;
	}
	else return false;
}

// ----------------------------------------------------------------------------
// This function determines the probability to move in a cartesian direction
// based on the potential.  It returns a 1,2,3 or 4 representing the four directions
int DLA::findProba()
{
	// Find the probabilities to move
	// p(x) = (1/c)exp[-beta(V(x+dx)-V(x))]
	float factor1=exp(beta*(grid[y-1][x] - grid[y][x]));
	float factor2=exp(beta*(grid[y+1][x] - grid[y][x]));
	float factor3=exp(beta*(grid[y][x-1] - grid[y][x]));
	float factor4=exp(beta*(grid[y][x+1] - grid[y][x]));
	float c = factor1+factor2+factor3+factor4;

	// Get a random number in [0,c]
	float z = getRandy(c);

	if ( z <= factor1 )	// up
		return 1;
	else if ( (z > factor1) && (z <= factor1+factor2) ) // down
		return 2;
	else if ( (z > factor1+factor2) && (z <= factor1+factor2+factor3) ) // left
		return 3;
	else if ( z > factor1+factor2+factor3 ) // right
		return 4;
	else cout << "Invalid Direction \n";
		return 0;
}

// ----------------------------------------------------------------------------
// calculates the potentials using an averaging technique
void DLA::calcPot()
{
	for (int i=0; i<sqrt(r); i++)
	for (int x=1; x<n-1; x++)
		for (int y=1; y<n-1; y++)
			if (grid[y][x] != potential)		
			{
				double sum = grid[y][x+1]+grid[y][x-1]+grid[y-1][x]+grid[y+1][x];
				grid[y][x] = sum/4;
			}
}

//-----------------------------------------------------------------------------
// Test Driver
void main()
{
	// Create DLA simulation
//	int dSpawn,dKill,modVal;
	double beta;
	double potential;
/*	cout << "dSpawn? (r+dSpawn = spawn radius) ";
	cin >> dSpawn;
	cout << "dKill? (r+dKill = kill radius) ";
	cin >> dKill;
	cout << "What's the mod value? ";
	cin >> modVal;
*/	cout << "What's the potential? ";
	cin >> potential;
	cout << "What's beta? ";
	cin >> beta;

	DLA Sim(300,5,20,potential,500,beta);
	Sim.simulate();

	Sim.potentialValsOut();
	Sim.gridOut();

	cout << "\a";

}